/// \file get_option_payoffs.cu
/// \brief Option pricing code for model proposed by Christoffersen et al (2012, JFE) 

#include "rng.c"

const int CALL = 0;
const int PUT  = 1;
const int JMAX = 10;
const double SQRT2PI = 2.506628274631;        // sqrt(2*pi)


/// Compute payoff of option
__device__ 
double get_payoff( double S,           ///< (in) stock price
                   double K,           ///< (in) strike price
                   int   option_type   ///< (in) Option types (0=call, 1=put)
               ) {
    switch (option_type) {
        case CALL:
            return ( S>K ? S-K : 0.0 );
        case PUT:
            return ( S>K ? 0.0 : K-S );
    }
    return 0.0;
}



__device__
double normal_pdf( double x, double mu, double sd )
{
    double z = (x-mu)/sd;
    return exp( -0.5*z*z ) / ( sd * SQRT2PI );
}



///    payoff_d,     (out) simulated payoffs                                   
///    state_d,      (in/out) Latent states (volatility and jump intensity)    
///    seed_d,       (in/out) RNG seeds                                        
///    parms,        (in) Model parameters                                     
///    strike,       (in) Strike prices                                        
///    option_type,  (in) 0=call, 1=put                                        
///    noption,      (in) Number of options to be priced                       
///    T,            (in) Time to maturity                                     
///    nsim          (in) Number of simulated paths to use                     
                                                                                 




__global__
void get_option_payoffs( double        * payoff_d,    
                         double        * state_d,     
                         unsigned int  * seed_d,      
                         const double  * parms,       
                         const double  * strike,      
                         const int     * option_type, 
                         int             noption,     
                         double          T,           
                         int             nsim         
                      )

{
    int tid = blockIdx.x*blockDim.x + threadIdx.x;
    int nthreads = blockDim.x*gridDim.x;

    double r      = parms[ 0];
    double lamz   = parms[ 1];
    double az     = parms[ 2];
    double bz     = parms[ 3];
    double cz     = parms[ 4];
    double dz     = parms[ 5];
    double ez     = parms[ 6];
    double wz     = parms[ 7];
    double lamy   = parms[ 8];
    double ay     = parms[ 9];
    double by     = parms[10];
    double cy     = parms[11];
    double dy     = parms[12];
    double ey     = parms[13];
    double wy     = parms[14];
    double muJ    = parms[15];
    double sigJ   = parms[16];
    double xi     = exp( muJ + 0.5* sigJ*sigJ ) - 1.0;

    for ( int i=tid; i<nsim; i+=nthreads) {

        unsigned int seed[4];
        get_seed( seed, seed_d, i, nsim ); 

        double logS = state_d[0*nsim+i];  
        double hz   = state_d[1*nsim+i];  
        double hy   = state_d[2*nsim+i]; 

        for (int t=0; t<T; t++) {
            double R, m, z, y, z1, z2, u, nj, pj;
            m = r + (lamz - 0.5)*hz + (lamy - xi)*hy;

            random_normal( &z1, &z2, seed );
            z = sqrt(hz)*z1;

            u  = random_uniform( seed );
            nj = JMAX;
            for ( int j=0; j<JMAX; j++) {
                u -= pj = (j==0 ? exp(-hy) : pj*hy/j );
                if (u<=0) {
                    nj = j;
                    break;
                }
            }    
            y = nj*muJ + sqrt(nj*sigJ*sigJ)*z2;
            R = m + y + z;                              // log return
            logS += R;                                  // log price

            hy = wy + by*hy + ay/hz*(z-cy*hz)*(z-cy*hz) + dy*(y - ey)*(y - ey );
            hz = wz + bz*hz + az/hz*(z-cz*hz)*(z-cz*hz) + dz*(y - ez)*(y - ez ); 

            if ( hz < 0.000004 )  { hz = 0.000004; }
            if ( hy < 0.00 )      { hy = 0.00; } 
            if ( hy > 1.00 )      { hy = 1.00; }
        }    

        for (int j=0; j<noption; j++) {
            payoff_d[j*nsim+i] = get_payoff( exp(logS), strike[j], option_type[j] );
        }
        state_d[0*nsim+i] = logS;  
        state_d[1*nsim+i] = hz;  
        state_d[2*nsim+i] = hy; 
        put_seed( seed, seed_d, i, nsim );
    }
}
