% Option pricing model, Christoffesen et al (2012, JFE)


clear all
nsim    = 500*1000;
logS0   = 0.0;         % log of initial stock price
theta   = get_theta();
hz0     = 0.0001;      % initial garch state
hy0     = 0.01;        % initial jump intensity

time_to_maturity  = [30  60 120 240];
strike            = [ 0.70 : .02 : 1.30];
type              = (strike<1);    % Always use out-of-the-money options
                                   % Call=0, PUT=1


% Seed for RNG
seed = gpuArray.randi( [1024 2^30], nsim, 4, 'uint32' );

% Load and configure kernel
kernel = parallel.gpu.CUDAKernel( 'get_option_payoffs.ptx', ...
                                  'get_option_payoffs.cu', ...
                                  'get_option_payoffs' );
kernel.ThreadBlockSize = 256;
kernel.GridSize        = 256;


noption = length(strike);
nperiod = length(time_to_maturity);

s_save       = cell(nperiod,1);
p_save       = cell(nperiod,1);
option_price = cell(nperiod,1);

tic
price = gpuArray( zeros( nsim, noption, 'double' ));
for j = 1:nperiod
    state = gpuArray( double( repmat( [logS0 hz0 hy0], nsim, 1 )));
    T = time_to_maturity(j);
    [ price state seed ] = feval( kernel, price, state, seed, theta, strike, type, noption, T, nsim );
    option_price{j} = gather( mean( price ) );      % Option prices
    p_save{j}       = gather( price );              % Simulated payoffs    
    s_save{j}       = gather( state );              % Simulated states
end
toc

tic
bsiv = gpuArray( zeros(noption,nperiod,'double' ));
for j = 1:nperiod
    for k=1:noption
        T = time_to_maturity(j);
        bsiv(k,j) = get_bsiv( option_price{j}(k), 1, strike(k), T, 0, type(k) );
    end
end
toc


figure
for i=1:4
    subplot(2,2,i)
    T = time_to_maturity(i);
    iv = squeeze(bsiv(:,i))*sqrt(250)*100;
    plot( strike, iv )
    title( sprintf( '%d days to maturity', time_to_maturity(i) ))
	xlabel( 'Moneyness' )
	ylabel( 'BSIV' )
    set(gca,'ylim',[10 20], 'xlim', [.68 1.32] );
end
print -depsc2 bsiv.eps



