/// \file simpleRandom.cu
/// \brief Demonstrates random number generation


#include "../include/mycuda.h"
#include "../include/mycuda_random.h"


using namespace mycuda;


///  Print fraction of z in each quantile

int verify( float * z, int n, float * q, float * d, int nq ) {
    printf( "             Expected   Realized\n" );
    printf( "================================\n" );
    for (int i=0; i<nq; i++) {
        int n_realized=0;
        for (int j=0; j<n; j++) {
            if ( ( i==0 || z[j]>q[i-1] ) && z[j]<=q[i] ) n_realized++;
        }
        printf( "Quantile %d: %9.6f  %9.6f\n", i, d[i], float(n_realized)/n );
    }
    return 0;
}




int main () {

    const int blocksize = 64;
    const int gridsize  = 64;
    const int n         = 1<<20;
    const int seed      = 27853;

    float         * z     = mapped_malloc<float>(n);
    curandState_t * state = device_malloc<curandState>(blocksize*gridsize);
    mycuda_random::random_init <<< gridsize, blocksize >>> ( state, seed, blocksize*gridsize );



    // Generate some uniform randoms
    mycuda_random::random_uniform <<< gridsize, blocksize >>> ( z, state, n );
    cudaDeviceSynchronize();

    // Print a few...
    printf( "\n\nUniform\n==============\n" );
    for (int i=0; i<50; i++) printf( "%8.4f\n", z[i] );

    // Check quantiles
    float q[10], d[10];
    for (int i=0;i<10;i++) { 
        q[i] = (i+1)*0.10;
        d[i] = 0.10;
    }
    verify( z, n, q, d, 10 );


    // Generate some normal randoms
    mycuda_random::random_normal <<< gridsize, blocksize >>> ( z, state, n );
    cudaDeviceSynchronize();

    // Print a few...
    printf( "\n\nNormal\n==============\n" );
    for (int i=0; i<50; i++) printf( "%8.4f\n", z[i] );

    // Check quantiles
    for (int i=0;i<10;i++) { 
        q[i] = mycuda_random::norm_cdf_inv( (i+1)*0.10f );
        d[i] = 0.10;
    }
    verify( z, n, q, d, 10 );
}






