#!/bin/bash

### Run as root.
### Edit mail-status.sh to include email address to which status reports are sent. 
### Edit this script as desired.
### Reboot after completion (to restart sshd and set environment variables).

PROGNAME=$(basename $0)

function error_exit {
	echo "${PROGNAME}: Exiting with error code ${1}">&2
	exit 1
}                                                                                         



echo config.sh: Checking that script is running as root and email address is updated
### Check that script is running as root
[ $(id -u) -eq 0 ]                   || error_exit 1

### Check that the email address in mail-status.sh has been changed from EMAIL@ADDRESS to something useful
grep EMAIL@ADDRESS mail-status.sh    && error_exit 2




echo config.sh: Updating system ...
yum -q -y update                    || error_exit 3  



echo config.sh: Doing some preliminary configuration...
yum install -y -q mc ctags git      || error_exit 4
cp prompt.sh /etc/profile.d         || error_exit 5



echo config.sh: setting up public key authorization
mkdir /etc/skel/.ssh                || error_exit 6
cp gpu_workshop.pub /etc/skel/.ssh/authorized_keys
cat gpu_workshop.pub >> ~/.ssh/authorized_keys



#echo config.sh: Enabling password based authorization for guest accounts ...
cp sshd_config /etc/ssh             || error_exit 7




echo config.sh: Setting up status reports ...
yum install -y -q mailx                                       || error_exit 31
cp mail-status.sh /etc/init.d                                 || error_exit 33
ln -s /etc/init.d/mail-status.sh /etc/cron.hourly             || error_exit 34
ln -s /etc/init.d/mail-status.sh /etc/rc3.d/S95mail-status    || error_exit 35
ln -s /etc/init.d/mail-status.sh /etc/rc0.d/K05mail-status    || error_exit 36
ln -s /etc/init.d/mail-status.sh /etc/rc6.d/K05mail-status    || error_exit 37

### Send status immediately (for testing) and set lock
./mail-status.sh                                              || error_exit 38




echo config.sh: Installing cuda ...
CUDA_INSTALLER=cuda_5.0.35_linux_64_rhel6.x-1.run
CUDA_DOWNLOAD_DIR=http://developer.download.nvidia.com/compute/cuda/5_0/rel-update-1/installers

yum -q -y install freeglut-devel                      || error_exit 41
wget ${CUDA_DOWNLOAD_DIR}/${CUDA_INSTALLER}           || error_exit 42
chmod u+x ${CUDA_INSTALLER}                           || error_exit 43

### Don't update driver.  Already installed version is more recent (Check installed version using nvidia-smi)
./${CUDA_INSTALLER} --toolkit --silent                || error_exit 44
./${CUDA_INSTALLER} --samples --silent                || error_exit 45
ln -s /usr/local/cuda-5.0 /usr/local/cuda             || error_exit 46

### Disable originally installed version          
mkdir /etc/profile.d.orig && mv /etc/profile.d/cudatoolkit* /etc/profile.d.orig                      
mkdir /etc/ld.so.conf.d.orig && mv /etc/ld.so.conf.d/cudatoolkit* /etc/ld.so.conf.d.orig
mv /opt/nvidia/cuda /usr/local/cuda-4.2             

### Setup environment
cp cuda.sh       /etc/profile.d                       || error_exit 47 

### LD_LIBRARY_PATH needed for nvprof, so don't bother with this, even though this is the better way to do it
#cp cuda.conf     /etc/ld.so.conf.d                    || error_exit 48 
#ldconfig                                              || error_exit 49




echo config.sh: Installing accelereyes ...
wget http://quantosanalytics.org/garland/gpu_workshop/ArrayFire-linux-1.9-20121120.zip || error_exit 51 
unzip -q -d /usr/local ArrayFire-linux-1.9-20121120.zip                           || error_exit 52
cp arrayfire.sh      /etc/profile.d                                               || error_exit 53   
cp arrayfire.conf    /etc/ld.so.conf.d                                            || error_exit 54 
ldconfig                                                                          || error_exit 55




echo config.sh: Installing pycuda ...
yum -y -q install  git ctags numpy scipy python-matplotlib ipython \
                   python-pandas sympy python-nose boost boost-python        || error_exit 61
easy_install Mako                                                            || error_exit 62
git clone http://git.tiker.net/trees/pycuda.git                              || error_exit 63
cd pycuda
./configure.py --cuda-root=/usr/local/cuda \
               --cudadrv-lib-dir=/usr/lib64/nvidia \
               --boost-python-libname=boost_python-mt \
               --boost-thread-libname=boost_thread-mt        || error_exit 64     
make                                                         || error_exit 65
make install                                                 || error_exit 66
cd ..




echo config.sh: Getting samples ...
cp -r /usr/local/cuda/samples         ~/cuda_examples                       || error_exit 73
cp -r /usr/local/arrayfire/examples   ~/arrayfire_examples                  || error_exit 74 
chown -R ec2-user: ~/cuda_examples                                          || error_exit 75
chown -R ec2-user: ~/arrayfire_examples                                     || error_exit 76

wget http://quantosanalytics.org/garland/gpu_workshop/code.zip                 || error_exit 91
unzip -q code.zip -d ..                                                        || error_exit 94
chown -R ec2-user: ../code                                                     || error_exit 95





echo config.sh: Setting up /etc/skel for user accounts ...
wget http://quantosanalytics.org/garland/gpu_workshop/arrayfire_examples.zip   || error_exit 92
wget http://quantosanalytics.org/garland/gpu_workshop/cuda_samples.zip         || error_exit 93
cp code.zip                /etc/skel                                           || error_exit 96
cp cuda_samples.zip        /etc/skel                                           || error_exit 97
cp arrayfire_examples.zip  /etc/skel                                           || error_exit 98





echo "\n\n====================================================================="
echo "            Please, log out and then back in to get updated environment"
echo "              To reboot:   sudo shutdown -r now"
echo "              To shutdown: sudo shutdown -h now"
echo "====================================================================="
